/** @file
  Register names for PCH SATA controllers

  Conventions:

  - Prefixes:
    Definitions beginning with "R_" are registers
    Definitions beginning with "B_" are bits within registers
    Definitions beginning with "V_" are meaningful values within the bits
    Definitions beginning with "S_" are register sizes
    Definitions beginning with "N_" are the bit position
  - In general, PCH registers are denoted by "_PCH_" in register names
  - Registers / bits that are different between PCH generations are denoted by
    "_PCH_[generation_name]_" in register/bit names.
  - Registers / bits that are specific to PCH-H denoted by "_H_" in register/bit names.
    Registers / bits that are specific to PCH-LP denoted by "_LP_" in register/bit names.
    e.g., "_PCH_H_", "_PCH_LP_"
    Registers / bits names without _H_ or _LP_ apply for both H and LP.
  - Registers / bits that are different between SKUs are denoted by "_[SKU_name]"
    at the end of the register/bit names
  - Registers / bits of new devices introduced in a PCH generation will be just named
    as "_PCH_" without [generation_name] inserted.

 @copyright
  INTEL CONFIDENTIAL
  Copyright 1999 - 2017 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#ifndef _PCH_REGS_SATA_H_
#define _PCH_REGS_SATA_H_

//
//  SATA Controller Registers (D23:F0)
//
#define PCI_DEVICE_NUMBER_PCH_SATA          23
#define PCI_FUNCTION_NUMBER_PCH_SATA        0
#define V_PCH_SATA_VENDOR_ID                V_PCH_INTEL_VENDOR_ID

//
//  SKL PCH-LP SATA Device ID's
//
#define V_PCH_LP_SATA_DEVICE_ID_M_AHCI         0x9D03  ///< SATA Controller (AHCI) - Mobile
#define V_PCH_LP_SATA_DEVICE_ID_M_RAID         0x9D05  ///< SATA Controller (RAID 0/1/5/10) - NOT premium - Mobile
#define V_PCH_LP_SATA_DEVICE_ID_M_RAID_ALTDIS  0x282A  ///< SATA Controller (RAID 0/1/5/10) - NOT premium - Mobile - Alternate ID
#define V_PCH_LP_SATA_DEVICE_ID_M_RAID_PREM    0x9D07  ///< SATA Controller (RAID 0/1/5/10) - premium - Mobile
#define V_PCH_LP_SATA_DEVICE_ID_M_RAID_RRT     0x9D0F  ///< SATA Controller (RAID 1/RRT Only) - Mobile

//
//  PCH-H SATA Device ID's
//
#define V_PCH_H_SATA_DEVICE_ID_D_RAID_ALTDIS   0x2822  ///< SATA Controller (RAID 0/1/5/10) - premium - Alternate ID
#define V_PCH_H_SATA_DEVICE_ID_D_RAID_RSTE     0x2826  ///< SATA Controller (RAID 0/1/5/10) - RSTe of Server SKU

//
//  SKL PCH-H SATA Device ID's
//
#define V_SKL_PCH_H_SATA_DEVICE_ID_D_AHCI          0xA102  ///< SATA Controller (AHCI)
#define V_SKL_PCH_H_SATA_DEVICE_ID_D_AHCI_A0       0xA103  ///< SATA Controller (AHCI) - SPTH A0
#define V_SKL_PCH_H_SATA_DEVICE_ID_D_RAID          0xA105  ///< SATA Controller (RAID 0/1/5/10) - NOT premium
#define V_SKL_PCH_H_SATA_DEVICE_ID_D_RAID_PREM     0xA107  ///< SATA Controller (RAID 0/1/5/10) - premium
#define V_SKL_PCH_H_SATA_DEVICE_ID_D_RAID_RRT      0xA10F  ///< SATA Controller (RAID 1/RRT Only)

//
//  KBL PCH-H SATA Device ID's
//
#define V_KBL_PCH_H_SATA_DEVICE_ID_D_AHCI          0xA282  ///< SATA Controller (AHCI)
#define V_KBL_PCH_H_SATA_DEVICE_ID_D_RAID          0xA284  ///< SATA Controller (RAID 0/1/5/10) - NOT premium
#define V_KBL_PCH_H_SATA_DEVICE_ID_D_RAID_PREM     0xA286  ///< SATA Controller (RAID 0/1/5/10) - premium
#define V_KBL_PCH_H_SATA_DEVICE_ID_D_OPTANE        0xA28E  ///< SATA Controller (RAID 1/RRT Only)


//
//  SATA Controller common Registers
//
#define V_PCH_SATA_SUB_CLASS_CODE_AHCI      0x06
#define V_PCH_SATA_SUB_CLASS_CODE_RAID      0x04
#define R_PCH_SATA_AHCI_BAR                 0x24
#define B_PCH_SATA_AHCI_BAR_BA              0xFFFFF800
#define V_PCH_SATA_AHCI_BAR_LENGTH          0x800
#define N_PCH_SATA_AHCI_BAR_ALIGNMENT       11
#define V_PCH_SATA_AHCI_BAR_LENGTH_512K     0x80000
#define N_PCH_SATA_AHCI_BAR_ALIGNMENT_512K  19
#define B_PCH_SATA_AHCI_BAR_PF              BIT3
#define B_PCH_SATA_AHCI_BAR_TP              (BIT2 | BIT1)
#define B_PCH_SATA_AHCI_BAR_RTE             BIT0
#define R_PCH_SATA_PID                      0x70
#define B_PCH_SATA_PID_NEXT                 0xFF00
#define V_PCH_SATA_PID_NEXT_0               0xB000
#define V_PCH_SATA_PID_NEXT_1               0xA800
#define B_PCH_SATA_PID_CID                  0x00FF
#define R_PCH_SATA_PC                       0x72
#define S_PCH_SATA_PC                       2
#define B_PCH_SATA_PC_PME                   (BIT15 | BIT14 | BIT13 | BIT12 | BIT11)
#define V_PCH_SATA_PC_PME_0                 0x0000
#define V_PCH_SATA_PC_PME_1                 0x4000
#define B_PCH_SATA_PC_D2_SUP                BIT10
#define B_PCH_SATA_PC_D1_SUP                BIT9
#define B_PCH_SATA_PC_AUX_CUR               (BIT8 | BIT7 | BIT6)
#define B_PCH_SATA_PC_DSI                   BIT5
#define B_PCH_SATA_PC_PME_CLK               BIT3
#define B_PCH_SATA_PC_VER                   (BIT2 | BIT1 | BIT0)
#define R_PCH_SATA_PMCS                     0x74
#define B_PCH_SATA_PMCS_PMES                BIT15
#define B_PCH_SATA_PMCS_PMEE                BIT8
#define B_PCH_SATA_PMCS_NSFRST              BIT3
#define V_PCH_SATA_PMCS_NSFRST_1            0x01
#define V_PCH_SATA_PMCS_NSFRST_0            0x00
#define B_PCH_SATA_PMCS_PS                  (BIT1 | BIT0)
#define V_PCH_SATA_PMCS_PS_3                0x03
#define V_PCH_SATA_PMCS_PS_0                0x00
#define R_PCH_SATA_MID                      0x80
#define B_PCH_SATA_MID_NEXT                 0xFF00
#define B_PCH_SATA_MID_CID                  0x00FF
#define R_PCH_SATA_MC                       0x82
#define B_PCH_SATA_MC_C64                   BIT7
#define B_PCH_SATA_MC_MME                   (BIT6 | BIT5 | BIT4)
#define V_PCH_SATA_MC_MME_4                 0x04
#define V_PCH_SATA_MC_MME_2                 0x02
#define V_PCH_SATA_MC_MME_1                 0x01
#define V_PCH_SATA_MC_MME_0                 0x00
#define B_PCH_SATA_MC_MMC                   (BIT3 | BIT2 | BIT1)
#define V_PCH_SATA_MC_MMC_4                 0x04
#define V_PCH_SATA_MC_MMC_0                 0x00
#define B_PCH_SATA_MC_MSIE                  BIT0
#define V_PCH_SATA_MC_MSIE_1                0x01
#define V_PCH_SATA_MC_MSIE_0                0x00
#define R_PCH_SATA_MA                       0x84
#define B_PCH_SATA_MA                       0xFFFFFFFC
#define R_PCH_SATA_MD                       0x88
#define B_PCH_SATA_MD_MSIMD                 0xFFFF

//
// Sata Register for PCH-LP
//
#define R_PCH_LP_SATA_MAP                   0x90
#define B_PCH_LP_SATA_MAP_SPD               (BIT10 | BIT9 | BIT8)
#define N_PCH_LP_SATA_MAP_SPD               8
#define B_PCH_LP_SATA_MAP_SPD2              BIT10
#define B_PCH_LP_SATA_MAP_SPD1              BIT9
#define B_PCH_LP_SATA_MAP_SPD0              BIT8
#define B_PCH_LP_SATA_MAP_SMS_MASK          BIT6
#define N_PCH_LP_SATA_MAP_SMS_MASK          6
#define V_PCH_LP_SATA_MAP_SMS_AHCI          0x0
#define V_PCH_LP_SATA_MAP_SMS_RAID          0x1
#define R_PCH_LP_SATA_PCS                   0x92
#define B_PCH_LP_SATA_PCS_OOB_RETRY         BIT15
#define B_PCH_LP_SATA_PCS_P2P               BIT10
#define B_PCH_LP_SATA_PCS_P1P               BIT9
#define B_PCH_LP_SATA_PCS_P0P               BIT8
#define B_PCH_LP_SATA_PCS_PXE_MASK          (BIT2 | BIT1 | BIT0)
#define B_PCH_LP_SATA_PCS_P2E               BIT2
#define B_PCH_LP_SATA_PCS_P1E               BIT1
#define B_PCH_LP_SATA_PCS_P0E               BIT0
#define R_PCH_LP_SATA_SCLKGC                0x94
#define B_PCH_LP_SATA_SCLKGC_PCD            (BIT26 | BIT25 | BIT24)
#define B_PCH_LP_SATA_SCLKGC_PORT2_PCD      BIT26
#define B_PCH_LP_SATA_SCLKGC_PORT1_PCD      BIT25
#define B_PCH_LP_SATA_SCLKGC_PORT0_PCD      BIT24
#define R_PCH_LP_SATA_98                    0x98

//
// Sata Register for PCH-H
//
#define R_PCH_H_SATA_MAP                    0x90
#define B_PCH_H_SATA_MAP_SPD                0xFF0000
#define N_PCH_H_SATA_MAP_SPD                16
#define B_PCH_H_SATA_MAP_SPD7               BIT23
#define B_PCH_H_SATA_MAP_SPD6               BIT22
#define B_PCH_H_SATA_MAP_SPD5               BIT21
#define B_PCH_H_SATA_MAP_SPD4               BIT20
#define B_PCH_H_SATA_MAP_SPD3               BIT19
#define B_PCH_H_SATA_MAP_SPD2               BIT18
#define B_PCH_H_SATA_MAP_SPD1               BIT17
#define B_PCH_H_SATA_MAP_SPD0               BIT16
#define B_PCH_H_SATA_MAP_PCD                0xFF
#define B_PCH_H_SATA_MAP_PORT7_PCD          BIT7
#define B_PCH_H_SATA_MAP_PORT6_PCD          BIT6
#define B_PCH_H_SATA_MAP_PORT5_PCD          BIT5
#define B_PCH_H_SATA_MAP_PORT4_PCD          BIT4
#define B_PCH_H_SATA_MAP_PORT3_PCD          BIT3
#define B_PCH_H_SATA_MAP_PORT2_PCD          BIT2
#define B_PCH_H_SATA_MAP_PORT1_PCD          BIT1
#define B_PCH_H_SATA_MAP_PORT0_PCD          BIT0
#define R_PCH_H_SATA_PCS                    0x94
#define B_PCH_H_SATA_PCS_P7P                BIT23
#define B_PCH_H_SATA_PCS_P6P                BIT22
#define B_PCH_H_SATA_PCS_P5P                BIT21
#define B_PCH_H_SATA_PCS_P4P                BIT20
#define B_PCH_H_SATA_PCS_P3P                BIT19
#define B_PCH_H_SATA_PCS_P2P                BIT18
#define B_PCH_H_SATA_PCS_P1P                BIT17
#define B_PCH_H_SATA_PCS_P0P                BIT16
#define B_PCH_H_SATA_PCS_PXE_MASK           0xFF
#define B_PCH_H_SATA_PCS_P7E                BIT7
#define B_PCH_H_SATA_PCS_P6E                BIT6
#define B_PCH_H_SATA_PCS_P5E                BIT5
#define B_PCH_H_SATA_PCS_P4E                BIT4
#define B_PCH_H_SATA_PCS_P3E                BIT3
#define B_PCH_H_SATA_PCS_P2E                BIT2
#define B_PCH_H_SATA_PCS_P1E                BIT1
#define B_PCH_H_SATA_PCS_P0E                BIT0

#define R_PCH_SATA_SATAGC                   0x9C
#define B_PCH_H_SATA_SATAGC_SMS_MASK        BIT16
#define N_PCH_H_SATA_SATAGC_SMS_MASK        16
#define V_PCH_H_SATA_SATAGC_SMS_AHCI        0x0
#define V_PCH_H_SATA_SATAGC_SMS_RAID        0x1
#define B_PCH_SATA_SATAGC_AIE               BIT7
#define B_PCH_SATA_SATAGC_AIES              BIT6
#define B_PCH_SATA_SATAGC_MSS               (BIT4 | BIT3)
#define V_PCH_SATA_SATAGC_MSS_8K            0x2
#define N_PCH_SATA_SATAGC_MSS               3
#define B_PCH_SATA_SATAGC_ASSEL             (BIT2 | BIT1 | BIT0)

#define V_PCH_SATA_SATAGC_ASSEL_2K          0x0
#define V_PCH_SATA_SATAGC_ASSEL_16K         0x1
#define V_PCH_SATA_SATAGC_ASSEL_32K         0x2
#define V_PCH_SATA_SATAGC_ASSEL_64K         0x3
#define V_PCH_SATA_SATAGC_ASSEL_128K        0x4
#define V_PCH_SATA_SATAGC_ASSEL_256K        0x5
#define V_PCH_SATA_SATAGC_ASSEL_512K        0x6

#define R_PCH_SATA_SIRI                     0xA0
#define R_PCH_SATA_STRD                     0xA4
#define R_PCH_SATA_SIR_0C                   0x0C
#define R_PCH_SATA_SIR_50                   0x50
#define R_PCH_SATA_SIR_54                   0x54
#define R_PCH_SATA_SIR_58                   0x58
#define R_PCH_SATA_SIR_5C                   0x5C
#define R_PCH_SATA_SIR_60                   0x60
#define R_PCH_SATA_SIR_64                   0x64
#define R_PCH_SATA_SIR_68                   0x68
#define R_PCH_SATA_SIR_6C                   0x6C
#define R_PCH_SATA_SIR_70                   0x70
#define R_PCH_SATA_SIR_80                   0x80
#define R_PCH_SATA_SIR_84                   0x84
#define R_PCH_SATA_SIR_8C                   0x8C
#define R_PCH_SATA_SIR_90                   0x90
#define R_PCH_SATA_SIR_98                   0x98
#define R_PCH_SATA_SIR_9C                   0x9C
#define R_PCH_SATA_SIR_A0                   0xA0
#define R_PCH_SATA_SIR_A4                   0xA4
#define R_PCH_SATA_SIR_A8                   0xA8
#define R_PCH_SATA_SIR_C8                   0xC8
#define R_PCH_SATA_SIR_CC                   0xCC
#define R_PCH_SATA_SIR_D0                   0xD0
#define R_PCH_SATA_SIR_D4                   0xD4
#define B_PCH_SATA_STRD_DTA                 0xFFFFFFFF
#define R_PCH_SATA_CR0                      0xA8
#define B_PCH_SATA_CR0_MAJREV               0x00F00000
#define B_PCH_SATA_CR0_MINREV               0x000F0000
#define B_PCH_SATA_CR0_NEXT                 0x0000FF00
#define B_PCH_SATA_CR0_CAP                  0x000000FF
#define R_PCH_SATA_CR1                      0xAC
#define B_PCH_SATA_CR1_BAROFST              0xFFF0
#define B_PCH_SATA_CR1_BARLOC               0x000F
#define R_PCH_SATA_FLR_CID                  0xB0
#define B_PCH_SATA_FLR_CID_NEXT             0xFF00
#define B_PCH_SATA_FLR_CID                  0x00FF
#define V_PCH_SATA_FLR_CID_1                0x0009
#define V_PCH_SATA_FLR_CID_0                0x0013
#define R_PCH_SATA_FLR_CLV                  0xB2
#define B_PCH_SATA_FLR_CLV_FLRC_FLRCSSEL_0  BIT9
#define B_PCH_SATA_FLR_CLV_TXPC_FLRCSSEL_0  BIT8
#define B_PCH_SATA_FLR_CLV_VSCID_FLRCSSEL_0 0x00FF
#define B_PCH_SATA_FLR_CLV_VSCID_FLRCSSEL_1 0x00FF
#define V_PCH_SATA_FLR_CLV_VSCID_FLRCSSEL   0x0006
#define R_PCH_SATA_FLRC                     0xB4
#define B_PCH_SATA_FLRC_TXP                 BIT8
#define B_PCH_SATA_FLRC_INITFLR             BIT0
#define R_PCH_SATA_SP                       0xC0
#define B_PCH_SATA_SP                       0xFFFFFFFF
#define R_PCH_SATA_MXID                     0xD0
#define N_PCH_SATA_MXID_NEXT                8

#define R_PCH_SATA_BFCS                     0xE0
#define B_PCH_SATA_BFCS_P7BFI               BIT17
#define B_PCH_SATA_BFCS_P6BFI               BIT16
#define B_PCH_SATA_BFCS_P5BFI               BIT15
#define B_PCH_SATA_BFCS_P4BFI               BIT14
#define B_PCH_SATA_BFCS_P3BFI               BIT13
#define B_PCH_SATA_BFCS_P2BFI               BIT12
#define B_PCH_SATA_BFCS_P2BFS               BIT11
#define B_PCH_SATA_BFCS_P2BFF               BIT10
#define B_PCH_SATA_BFCS_P1BFI               BIT9
#define B_PCH_SATA_BFCS_P0BFI               BIT8
#define B_PCH_SATA_BFCS_BIST_FIS_T          BIT7
#define B_PCH_SATA_BFCS_BIST_FIS_A          BIT6
#define B_PCH_SATA_BFCS_BIST_FIS_S          BIT5
#define B_PCH_SATA_BFCS_BIST_FIS_L          BIT4
#define B_PCH_SATA_BFCS_BIST_FIS_F          BIT3
#define B_PCH_SATA_BFCS_BIST_FIS_P          BIT2
#define R_PCH_SATA_BFTD1                    0xE4
#define B_PCH_SATA_BFTD1                    0xFFFFFFFF
#define R_PCH_SATA_BFTD2                    0xE8
#define B_PCH_SATA_BFTD2                    0xFFFFFFFF

#define R_PCH_SATA_VS_CAP                   0xA4
#define B_PCH_SATA_VS_CAP_NRMBE             BIT0                            ///< NVM Remap Memory BAR Enable
#define B_PCH_SATA_VS_CAP_MSL               0x1FFE                          ///< Memory Space Limit
#define N_PCH_SATA_VS_CAP_MSL               1
#define V_PCH_SATA_VS_CAP_MSL               0x1EF                           ///< Memory Space Limit Field Value
#define B_PCH_SATA_VS_CAP_NRMO              0xFFF0000                       ///< NVM Remapped Memory Offset
#define N_PCH_SATA_VS_CAP_NRMO              16
#define V_PCH_SATA_VS_CAP_NRMO              0x10                            ///< NVM Remapped Memory Offset Field Value

//
// RST PCIe Storage Remapping Registers
//
#define R_PCH_RST_PCIE_STORAGE_RCR                 0x800                           ///< Remap Capability Register
#define B_PCH_RST_PCIE_STORAGE_RCR_NRS             (BIT2|BIT1|BIT0)                ///< Number of Remapping Supported
#define B_PCH_RST_PCIE_STORAGE_RCR_NRS_CR1         BIT0                            ///< Number of Remapping Supported (RST PCIe Storage Cycle Router #1)
#ifndef MINTREE_FLAG
#define R_PCH_RST_PCIE_STORAGE_AMXC                0x808                           ///< AHCI MSI-X Configuration
#define B_PCH_RST_PCIE_STORAGE_AMXC_AMXV           0x7FF                           ///< AHCI MSI-X Vector
#define R_PCH_RST_PCIE_STORAGE_CR1_DMBL            0x884                           ///< CR#1 Device Memory BAR Length
#define R_PCH_RST_PCIE_STORAGE_CR1_DMXC            0x888                           ///< CR#1 Device MSI-X Configuration
#define N_PCH_RST_PCIE_STORAGE_CR1_DMXC_MXEV       16
#endif //MINTREE_FLAG
#define R_PCH_RST_PCIE_STORAGE_SPR                 0x80C                           ///< Scratch Pad Register
#define R_PCH_RST_PCIE_STORAGE_CR1_DCC             0x880                           ///< CR#1 Device Class Code
#define N_PCH_RST_PCIE_STORAGE_CR1_DCC_SCC         8
#define N_PCH_RST_PCIE_STORAGE_CR1_DCC_BCC         16
#define B_PCH_RST_PCIE_STORAGE_CR1_DCC_DT          BIT31                           ///< Device Type
#define V_PCH_RST_PCIE_STORAGE_REMAP_CONFIG_CR     0x80                            ///< Remapped Configuration for RST PCIe Storage Cycle Router #n
#define V_PCH_RST_PCIE_STORAGE_REMAP_RP_OFFSET     0x100                           ///< Remapped Root Port Offset Value
#define R_PCH_RST_PCIE_STORAGE_CCFG                0x1D0                           ///< Port Configuration Register
#ifndef MINTREE_FLAG
#define R_PCH_RST_PCIE_STORAGE_GCR                 0x300                           ///< General Configuration Register
#define B_PCH_RST_PCIE_STORAGE_GCR_CRE             BIT0                            ///< RST PCIe Storage Cycle Router Enable
#define B_PCH_RST_PCIE_STORAGE_GCR_PLS             0x1FFFFE                        ///< PCIe Lane Selected Field
#define B_PCH_RST_PCIE_STORAGE_GCR_PCRTCGP         BIT28                           ///< PCIe Cycle Router Trunk Clock Gating Policy
#define B_PCH_RST_PCIE_STORAGE_GCR_PNCAIDL         BIT29                           ///< Configuration Access Index/Data Lockdown
#define B_PCH_RST_PCIE_STORAGE_GCR_RCL             BIT30                           ///< Remapping Configuration Lockdown
#define B_PCH_RST_PCIE_STORAGE_GCR_CREL            BIT31                           ///< RST PCIe Storage Cycle Router Enable Lockdown
#define R_PCH_RST_PCIE_STORAGE_GSR                 0x304                           ///< General Status Register
#define B_PCH_RST_PCIE_STORAGE_GSR_PLRC            0x1FFFFE                        ///< PCIe Lane Remap Capable Field
#define B_PCH_RST_PCIE_STORAGE_GSR_PCCS            BIT30                           ///< Port Configuration Check Status
#define B_PCH_RST_PCIE_STORAGE_GSR_PCCD            BIT31                           ///< Port Configuration Check Disable
#define R_PCH_RST_PCIE_STORAGE_CAIR                0x308                           ///< Configuration Access Index Register
#define R_PCH_RST_PCIE_STORAGE_CADR                0x30C                           ///< Configuration Access Data Register
#define R_PCH_RST_PCIE_STORAGE_MBRC                0x310                           ///< Memory BAR Remap Configuration
#define B_PCH_RST_PCIE_STORAGE_MBRC_TMB            (BIT19|BIT18|BIT17|BIT16)       ///< Memory BAR Remap Configuration Target Memory BAR
#define N_PCH_RST_PCIE_STORAGE_MBRC_TMB            16
#define V_PCH_RST_PCIE_STORAGE_MBRC_TMB_BAR0       0x4                             ///< Memory BAR Remap Configuration Target Memory BAR - BAR0
#define V_PCH_RST_PCIE_STORAGE_MBRC_TMB_BAR5       0x9                             ///< Memory BAR Remap Configuration Target Memory BAR - BAR5
#define B_PCH_RST_PCIE_STORAGE_MBRC_TT             BIT20                           ///< Remap Configuration Target Type
#define B_PCH_RST_PCIE_STORAGE_MBRC_RE             BIT31                           ///< Remap Configuration Remap Enable
#define R_PCH_RST_PCIE_STORAGE_IOBRSC              0x320       ///< I/O BAR Remap Source Configuration
#define B_PCH_RST_PCIE_STORAGE_IOBRSC_RE           BIT31       ///< I/O BAR Remap Source Configuration Remap Enable
#define R_PCH_RST_PCIE_STORAGE_AIDPCRC             0x338       ///< AHCI Index/Data Pair Capability Remap Configuration
#define B_PCH_RST_PCIE_STORAGE_AIDPCRC_TCSO        0xFF0000    ///< AHCI Index/Data Pair Capability Remap Configuration Target Capability Structure OffsBet
#define N_PCH_RST_PCIE_STORAGE_AIDPCRC_TCSO        16
#define B_PCH_RST_PCIE_STORAGE_AIDPCRC_RE          BIT31       ///< AHCI Index/Data Pair Capability Remap Configuration Remap Enable
#define R_PCH_RST_PCIE_STORAGE_MXCRC               0x33C       ///< MSI-X Capability Remap Configuration
#define B_PCH_RST_PCIE_STORAGE_MXCRCC_TCSO         0xFF0000    ///< MSI-X Capability Remap Configuration Target Capability Structure Offset
#define N_PCH_RST_PCIE_STORAGE_MXCRCC_TCSO         16
#define B_PCH_RST_PCIE_STORAGE_MXCRC_RE            BIT31       ///< MSI-X Capability Remap Configuration Remap Enable
#define R_PCH_RST_PCIE_STORAGE_MXTRC               0x340       ///< MSI-X Table Remap Configuration
#define B_PCH_RST_PCIE_STORAGE_MXTRC_TBIR          0x07        ///< MSI-X Table Remap Configuration Table BIR
#define B_PCH_RST_PCIE_STORAGE_MXTRC_TO            0xFFFFFFF8  ///< MSI-X Table Remap Configuration Table Offset
#define R_PCH_RST_PCIE_STORAGE_MXTBAR              0x344       ///< MSI-X Table Base Address Register
#define B_PCH_RST_PCIE_STORAGE_MXTBAR_TBAV         BIT0        ///< MSI-X Table Base Address Register Table Base Address Valid
#define R_PCH_RST_PCIE_STORAGE_MXPRC               0x348       ///< MSI-X PBA Remap Configuration
#define B_PCH_RST_PCIE_STORAGE_MXPRC_TBIR          0x07        ///< MSI-X PBA Remap Configuration Table BIR
#define B_PCH_RST_PCIE_STORAGE_MXPRC_TO            0xFFFFFFF8  ///< MSI-X PBA Remap Configuration Table Offset
#define R_PCH_RST_PCIE_STORAGE_MXPBAR              0x34C       ///< MSI-X PBA Base Address Register
#define B_PCH_RST_PCIE_STORAGE_MXPBAR_TBAV         BIT0        ///< MSI-X PBA Base Address Register Table Base Address Valid
#define R_PCH_RST_PCIE_STORAGE_NRDF                0x350       ///< NVM Remapping Device:Function
#define R_PCH_RST_PCIE_STORAGE_EGCR                0x354       ///< Extended General Configuration Register
#define B_PCH_RST_PCIE_STORAGE_EGCR_CRDCGE         BIT16       ///< RST PCIe Storage Cycle Router Dynamic Clock Gating Enable
#define B_PCH_RST_PCIE_STORAGE_EGCR_CRTCGE         BIT17       ///< RST PCIe Storage Cycle Router Trunk Clock Gating Enable
#define B_PCH_RST_PCIE_STORAGE_EGCR_ICAS           BIT19       ///< IOSF CLKREQ Assertion Select
#define B_PCH_RST_PCIE_STORAGE_EGCR_TSCAS          BIT20       ///< To SATA CLKREQ Assertion Select
#define R_PCH_RST_PCIE_STORAGE_SAPI                0x358       ///< Shadowed AHCI Ports Implemented
#define R_PCH_RST_PCIE_STORAGE_CRGC                0xFC0       ///< RST PCIe Storage Cycle Router Global Control
#define B_PCH_RST_PCIE_STORAGE_CRGC_CRAS           (BIT1|BIT0) ///< RST PCIe Storage Cycle Router Global Control Cycle Router Accessibility Select
#endif //MINTREE_FLAG

//
// AHCI BAR Area related Registers
//
#define R_PCH_SATA_AHCI_CAP                 0x0
#define B_PCH_SATA_AHCI_CAP_S64A            BIT31
#define B_PCH_SATA_AHCI_CAP_SCQA            BIT30
#define B_PCH_SATA_AHCI_CAP_SSNTF           BIT29
#define B_PCH_SATA_AHCI_CAP_SIS             BIT28 ///< Supports Interlock Switch
#define B_PCH_SATA_AHCI_CAP_SSS             BIT27 ///< Supports Stagger Spin-up
#define B_PCH_SATA_AHCI_CAP_SALP            BIT26
#define B_PCH_SATA_AHCI_CAP_SAL             BIT25
#define B_PCH_SATA_AHCI_CAP_SCLO            BIT24 ///< Supports Command List Override
#define B_PCH_SATA_AHCI_CAP_ISS_MASK        (BIT23 | BIT22 | BIT21 | BIT20)
#define N_PCH_SATA_AHCI_CAP_ISS             20    ///< Interface Speed Support
#define V_PCH_SATA_AHCI_CAP_ISS_1_5_G       0x01
#define V_PCH_SATA_AHCI_CAP_ISS_3_0_G       0x02
#define V_PCH_SATA_AHCI_CAP_ISS_6_0_G       0x03
#define B_PCH_SATA_AHCI_CAP_SNZO            BIT19
#define B_PCH_SATA_AHCI_CAP_SAM             BIT18
#define B_PCH_SATA_AHCI_CAP_PMS             BIT17 ///< Supports Port Multiplier
#define B_PCH_SATA_AHCI_CAP_PMD             BIT15 ///< PIO Multiple DRQ Block
#define B_PCH_SATA_AHCI_CAP_SSC             BIT14
#define B_PCH_SATA_AHCI_CAP_PSC             BIT13
#define B_PCH_SATA_AHCI_CAP_NCS             0x1F00
#define B_PCH_SATA_AHCI_CAP_CCCS            BIT7
#define B_PCH_SATA_AHCI_CAP_EMS             BIT6
#define B_PCH_SATA_AHCI_CAP_SXS             BIT5  ///< External SATA is supported
#define B_PCH_SATA_AHCI_CAP_NPS             0x001F

#define R_PCH_SATA_AHCI_GHC                 0x04
#define B_PCH_SATA_AHCI_GHC_AE              BIT31
#define B_PCH_SATA_AHCI_GHC_MRSM            BIT2
#define B_PCH_SATA_AHCI_GHC_IE              BIT1
#define B_PCH_SATA_AHCI_GHC_HR              BIT0

#define R_PCH_SATA_AHCI_IS                  0x08
#define B_PCH_SATA_AHCI_IS_PORT7            BIT7
#define B_PCH_SATA_AHCI_IS_PORT6            BIT6
#define B_PCH_SATA_AHCI_IS_PORT5            BIT5
#define B_PCH_SATA_AHCI_IS_PORT4            BIT4
#define B_PCH_SATA_AHCI_IS_PORT3            BIT3
#define B_PCH_SATA_AHCI_IS_PORT2            BIT2
#define B_PCH_SATA_AHCI_IS_PORT1            BIT1
#define B_PCH_SATA_AHCI_IS_PORT0            BIT0
#define R_PCH_SATA_AHCI_PI                  0x0C
#define B_PCH_H_SATA_PORT_MASK              0xFF
#define B_PCH_LP_SATA_PORT_MASK             0x03
#define B_PCH_SATA_PORT7_IMPLEMENTED        BIT7
#define B_PCH_SATA_PORT6_IMPLEMENTED        BIT6
#define B_PCH_SATA_PORT5_IMPLEMENTED        BIT5
#define B_PCH_SATA_PORT4_IMPLEMENTED        BIT4
#define B_PCH_SATA_PORT3_IMPLEMENTED        BIT3
#define B_PCH_SATA_PORT2_IMPLEMENTED        BIT2
#define B_PCH_SATA_PORT1_IMPLEMENTED        BIT1
#define B_PCH_SATA_PORT0_IMPLEMENTED        BIT0
#define R_PCH_SATA_AHCI_VS                  0x10
#define B_PCH_SATA_AHCI_VS_MJR              0xFFFF0000
#define B_PCH_SATA_AHCI_VS_MNR              0x0000FFFF
#define R_PCH_SATA_AHCI_EM_LOC              0x1C
#define B_PCH_SATA_AHCI_EM_LOC_OFST         0xFFFF0000
#define B_PCH_SATA_AHCI_EM_LOC_SZ           0x0000FFFF
#define R_PCH_SATA_AHCI_EM_CTRL             0x20
#define B_PCH_SATA_AHCI_EM_CTRL_ATTR_ALHD   BIT26
#define B_PCH_SATA_AHCI_EM_CTRL_ATTR_XMT    BIT25
#define B_PCH_SATA_AHCI_EM_CTRL_ATTR_SMB    BIT24
#define B_PCH_SATA_AHCI_EM_CTRL_SUPP_SGPIO  BIT19
#define B_PCH_SATA_AHCI_EM_CTRL_SUPP_SES2   BIT18
#define B_PCH_SATA_AHCI_EM_CTRL_SUPP_SAFTE  BIT17
#define B_PCH_SATA_AHCI_EM_CTRL_SUPP_LED    BIT16
#define B_PCH_SATA_AHCI_EM_CTRL_RST         BIT9
#define B_PCH_SATA_AHCI_EM_CTRL_CTL_TM      BIT8
#define B_PCH_SATA_AHCI_EM_CTRL_STS_MR      BIT0
#define R_PCH_SATA_AHCI_CAP2                0x24
#define B_PCH_SATA_AHCI_CAP2_DESO           BIT5
#define B_PCH_SATA_AHCI_CAP2_SADM           BIT4
#define B_PCH_SATA_AHCI_CAP2_SDS            BIT3
#define B_PCH_SATA_AHCI_CAP2_APST           BIT2  ///< Automatic Partial to Slumber Transitions
#define R_PCH_SATA_AHCI_VSP                 0xA0
#define B_PCH_SATA_AHCI_VSP_SLPD            BIT0
#define R_PCH_SATA_AHCI_RSTF                0xC8  ///< RST Feature Capabilities
#define B_PCH_SATA_AHCI_RSTF_CPU_STORAGE    BIT14
#define B_PCH_SATA_AHCI_RSTF_OMA            BIT13
#define B_PCH_SATA_AHCI_RSTF_LEGACY         BIT12
#define B_PCH_SATA_AHCI_RSTF_OUD            (BIT11 | BIT10)
#define N_PCH_SATA_AHCI_RSTF_OUD            10
#define B_PCH_SATA_AHCI_RSTF_SEREQ          BIT9
#define B_PCH_SATA_AHCI_RSTF_IROES          BIT8
#define B_PCH_SATA_AHCI_RSTF_LEDL           BIT7
#define B_PCH_SATA_AHCI_RSTF_HDDLK          BIT6
#define B_PCH_SATA_AHCI_RSTF_IRSTOROM       BIT5
#define B_PCH_SATA_AHCI_RSTF_RSTE           BIT4
#define B_PCH_SATA_AHCI_RSTF_R5E            BIT3
#define B_PCH_SATA_AHCI_RSTF_R10E           BIT2
#define B_PCH_SATA_AHCI_RSTF_R1E            BIT1
#define B_PCH_SATA_AHCI_RSTF_R0E            BIT0
#define B_PCH_SATA_AHCI_RSTF_LOWBYTES       0x1FF
#define R_PCH_SATA_AHCI_P0CLB               0x100
#define R_PCH_SATA_AHCI_P1CLB               0x180
#define R_PCH_SATA_AHCI_P2CLB               0x200
#define R_PCH_SATA_AHCI_P3CLB               0x280
#define R_PCH_SATA_AHCI_P4CLB               0x300
#define R_PCH_SATA_AHCI_P5CLB               0x380
#define R_PCH_SATA_AHCI_P6CLB               0x400
#define R_PCH_SATA_AHCI_P7CLB               0x480
#define B_PCH_SATA_AHCI_PXCLB               0xFFFFFC00
#define R_PCH_SATA_AHCI_P0CLBU              0x104
#define R_PCH_SATA_AHCI_P1CLBU              0x184
#define R_PCH_SATA_AHCI_P2CLBU              0x204
#define R_PCH_SATA_AHCI_P3CLBU              0x284
#define R_PCH_SATA_AHCI_P4CLBU              0x304
#define R_PCH_SATA_AHCI_P5CLBU              0x384
#define R_PCH_SATA_AHCI_P6CLBU              0x404
#define R_PCH_SATA_AHCI_P7CLBU              0x484
#define B_PCH_SATA_AHCI_PXCLBU              0xFFFFFFFF
#define R_PCH_SATA_AHCI_P0FB                0x108
#define R_PCH_SATA_AHCI_P1FB                0x188
#define R_PCH_SATA_AHCI_P2FB                0x208
#define R_PCH_SATA_AHCI_P3FB                0x288
#define R_PCH_SATA_AHCI_P4FB                0x308
#define R_PCH_SATA_AHCI_P5FB                0x388
#define R_PCH_SATA_AHCI_P6FB                0x408
#define R_PCH_SATA_AHCI_P7FB                0x488
#define B_PCH_SATA_AHCI_PXFB                0xFFFFFF00
#define R_PCH_SATA_AHCI_P0FBU               0x10C
#define R_PCH_SATA_AHCI_P1FBU               0x18C
#define R_PCH_SATA_AHCI_P2FBU               0x20C
#define R_PCH_SATA_AHCI_P3FBU               0x28C
#define R_PCH_SATA_AHCI_P4FBU               0x30C
#define R_PCH_SATA_AHCI_P5FBU               0x38C
#define R_PCH_SATA_AHCI_P6FBU               0x40C
#define R_PCH_SATA_AHCI_P7FBU               0x48C
#define B_PCH_SATA_AHCI_PXFBU               0xFFFFFFFF
#define R_PCH_SATA_AHCI_P0IS                0x110
#define R_PCH_SATA_AHCI_P1IS                0x190
#define R_PCH_SATA_AHCI_P2IS                0x210
#define R_PCH_SATA_AHCI_P3IS                0x290
#define R_PCH_SATA_AHCI_P4IS                0x310
#define R_PCH_SATA_AHCI_P5IS                0x390
#define R_PCH_SATA_AHCI_P6IS                0x410
#define R_PCH_SATA_AHCI_P7IS                0x490
#define B_PCH_SATA_AHCI_PXIS_CPDS           BIT31
#define B_PCH_SATA_AHCI_PXIS_TFES           BIT30
#define B_PCH_SATA_AHCI_PXIS_HBFS           BIT29
#define B_PCH_SATA_AHCI_PXIS_HBDS           BIT28
#define B_PCH_SATA_AHCI_PXIS_IFS            BIT27
#define B_PCH_SATA_AHCI_PXIS_INFS           BIT26
#define B_PCH_SATA_AHCI_PXIS_OFS            BIT24
#define B_PCH_SATA_AHCI_PXIS_IPMS           BIT23
#define B_PCH_SATA_AHCI_PXIS_PRCS           BIT22
#define B_PCH_SATA_AHCI_PXIS_DIS            BIT7
#define B_PCH_SATA_AHCI_PXIS_PCS            BIT6
#define B_PCH_SATA_AHCI_PXIS_DPS            BIT5
#define B_PCH_SATA_AHCI_PXIS_UFS            BIT4
#define B_PCH_SATA_AHCI_PXIS_SDBS           BIT3
#define B_PCH_SATA_AHCI_PXIS_DSS            BIT2
#define B_PCH_SATA_AHCI_PXIS_PSS            BIT1
#define B_PCH_SATA_AHCI_PXIS_DHRS           BIT0
#define R_PCH_SATA_AHCI_P0IE                0x114
#define R_PCH_SATA_AHCI_P1IE                0x194
#define R_PCH_SATA_AHCI_P2IE                0x214
#define R_PCH_SATA_AHCI_P3IE                0x294
#define R_PCH_SATA_AHCI_P4IE                0x314
#define R_PCH_SATA_AHCI_P5IE                0x394
#define R_PCH_SATA_AHCI_P6IE                0x414
#define R_PCH_SATA_AHCI_P7IE                0x494
#define B_PCH_SATA_AHCI_PXIE_CPDE           BIT31
#define B_PCH_SATA_AHCI_PXIE_TFEE           BIT30
#define B_PCH_SATA_AHCI_PXIE_HBFE           BIT29
#define B_PCH_SATA_AHCI_PXIE_HBDE           BIT28
#define B_PCH_SATA_AHCI_PXIE_IFE            BIT27
#define B_PCH_SATA_AHCI_PXIE_INFE           BIT26
#define B_PCH_SATA_AHCI_PXIE_OFE            BIT24
#define B_PCH_SATA_AHCI_PXIE_IPME           BIT23
#define B_PCH_SATA_AHCI_PXIE_PRCE           BIT22
#define B_PCH_SATA_AHCI_PXIE_DIE            BIT7
#define B_PCH_SATA_AHCI_PXIE_PCE            BIT6
#define B_PCH_SATA_AHCI_PXIE_DPE            BIT5
#define B_PCH_SATA_AHCI_PXIE_UFIE           BIT4
#define B_PCH_SATA_AHCI_PXIE_SDBE           BIT3
#define B_PCH_SATA_AHCI_PXIE_DSE            BIT2
#define B_PCH_SATA_AHCI_PXIE_PSE            BIT1
#define B_PCH_SATA_AHCI_PXIE_DHRE           BIT0
#define R_PCH_SATA_AHCI_P0CMD               0x118
#define R_PCH_SATA_AHCI_P1CMD               0x198
#define R_PCH_SATA_AHCI_P2CMD               0x218
#define R_PCH_SATA_AHCI_P3CMD               0x298
#define R_PCH_SATA_AHCI_P4CMD               0x318
#define R_PCH_SATA_AHCI_P5CMD               0x398
#define R_PCH_SATA_AHCI_P6CMD               0x418
#define R_PCH_SATA_AHCI_P7CMD               0x498
#define B_PCH_SATA_AHCI_PxCMD_ICC           (BIT31 | BIT30 | BIT29 | BIT28)
#define B_PCH_SATA_AHCI_PxCMD_MASK          (BIT27 | BIT26 | BIT22 | BIT21 | BIT19 | BIT18)
#define B_PCH_SATA_AHCI_PxCMD_ASP           BIT27
#define B_PCH_SATA_AHCI_PxCMD_ALPE          BIT26
#define B_PCH_SATA_AHCI_PxCMD_DLAE          BIT25
#define B_PCH_SATA_AHCI_PxCMD_ATAPI         BIT24
#define B_PCH_SATA_AHCI_PxCMD_APSTE         BIT23
#define B_PCH_SATA_AHCI_PxCMD_SUD           BIT1
#define R_PCH_SATA_AHCI_P0DEVSLP            0x144
#define R_PCH_SATA_AHCI_P1DEVSLP            0x1C4
#define R_PCH_SATA_AHCI_P2DEVSLP            0x244
#define R_PCH_SATA_AHCI_P3DEVSLP            0x2C4
#define R_PCH_SATA_AHCI_P4DEVSLP            0x344
#define R_PCH_SATA_AHCI_P5DEVSLP            0x3C4
#define R_PCH_SATA_AHCI_P6DEVSLP            0x444
#define R_PCH_SATA_AHCI_P7DEVSLP            0x4C4
#define B_PCH_SATA_AHCI_PxDEVSLP_DSP        BIT1
#define B_PCH_SATA_AHCI_PxDEVSLP_ADSE       BIT0
#define B_PCH_SATA_AHCI_PxDEVSLP_DITO_MASK  0x01FF8000
#define V_PCH_SATA_AHCI_PxDEVSLP_DITO_625   0x01388000
#define B_PCH_SATA_AHCI_PxDEVSLP_DM_MASK    0x1E000000
#define V_PCH_SATA_AHCI_PxDEVSLP_DM_16      0x1E000000
#define B_PCH_SATA_AHCI_PxCMD_ESP           BIT21 ///< Used with an external SATA device
#define B_PCH_SATA_AHCI_PxCMD_MPSP          BIT19 ///< Mechanical Switch Attached to Port
#define B_PCH_SATA_AHCI_PxCMD_HPCP          BIT18 ///< Hotplug capable
#define B_PCH_SATA_AHCI_PxCMD_CR            BIT15
#define B_PCH_SATA_AHCI_PxCMD_FR            BIT14
#define B_PCH_SATA_AHCI_PxCMD_ISS           BIT13
#define B_PCH_SATA_AHCI_PxCMD_CCS           0x00001F00
#define B_PCH_SATA_AHCI_PxCMD_FRE           BIT4
#define B_PCH_SATA_AHCI_PxCMD_CLO           BIT3
#define B_PCH_SATA_AHCI_PxCMD_POD           BIT2
#define B_PCH_SATA_AHCI_PxCMD_SUD           BIT1
#define B_PCH_SATA_AHCI_PxCMD_ST            BIT0
#define R_PCH_SATA_AHCI_P0TFD               0x120
#define R_PCH_SATA_AHCI_P1TFD               0x1A0
#define R_PCH_SATA_AHCI_P2TFD               0x220
#define R_PCH_SATA_AHCI_P3TFD               0x2A0
#define R_PCH_SATA_AHCI_P4TFD               0x320
#define R_PCH_SATA_AHCI_P5TFD               0x3A0
#define R_PCH_SATA_AHCI_P6TFD               0x420
#define B_PCH_SATA_AHCI_PXTFD_ERR           0x0000FF00
#define B_PCH_SATA_AHCI_PXTFD_STS           0x000000FF
#define R_PCH_SATA_AHCI_P0SIG               0x124
#define R_PCH_SATA_AHCI_P1SIG               0x1A4
#define R_PCH_SATA_AHCI_P2SIG               0x224
#define R_PCH_SATA_AHCI_P3SIG               0x2A4
#define R_PCH_SATA_AHCI_P4SIG               0x324
#define R_PCH_SATA_AHCI_P5SIG               0x3A4
#define R_PCH_SATA_AHCI_P6SIG               0x424
#define B_PCH_SATA_AHCI_PXSIG_LBA_HR        0xFF000000
#define B_PCH_SATA_AHCI_PXSIG_LBA_MR        0x00FF0000
#define B_PCH_SATA_AHCI_PXSIG_LBA_LR        0x0000FF00
#define B_PCH_SATA_AHCI_PXSIG_SCR           0x000000FF
#define R_PCH_SATA_AHCI_P0SSTS              0x128
#define R_PCH_SATA_AHCI_P1SSTS              0x1A8
#define R_PCH_SATA_AHCI_P2SSTS              0x228
#define R_PCH_SATA_AHCI_P3SSTS              0x2A8
#define R_PCH_SATA_AHCI_P4SSTS              0x328
#define R_PCH_SATA_AHCI_P5SSTS              0x3A8
#define R_PCH_SATA_AHCI_P6SSTS              0x428
#define B_PCH_SATA_AHCI_PXSSTS_IPM_0        0x00000000
#define B_PCH_SATA_AHCI_PXSSTS_IPM_1        0x00000100
#define B_PCH_SATA_AHCI_PXSSTS_IPM_2        0x00000200
#define B_PCH_SATA_AHCI_PXSSTS_IPM_6        0x00000600
#define B_PCH_SATA_AHCI_PXSSTS_SPD_0        0x00000000
#define B_PCH_SATA_AHCI_PXSSTS_SPD_1        0x00000010
#define B_PCH_SATA_AHCI_PXSSTS_SPD_2        0x00000020
#define B_PCH_SATA_AHCI_PXSSTS_SPD_3        0x00000030
#define B_PCH_SATA_AHCI_PXSSTS_DET_0        0x00000000
#define B_PCH_SATA_AHCI_PXSSTS_DET_1        0x00000001
#define B_PCH_SATA_AHCI_PXSSTS_DET_3        0x00000003
#define B_PCH_SATA_AHCI_PXSSTS_DET_4        0x00000004
#define R_PCH_SATA_AHCI_P0SCTL              0x12C
#define R_PCH_SATA_AHCI_P1SCTL              0x1AC
#define R_PCH_SATA_AHCI_P2SCTL              0x22C
#define R_PCH_SATA_AHCI_P3SCTL              0x2AC
#define R_PCH_SATA_AHCI_P4SCTL              0x32C
#define R_PCH_SATA_AHCI_P5SCTL              0x3AC
#define R_PCH_SATA_AHCI_P6SCTL              0x42C
#define B_PCH_SATA_AHCI_PXSCTL_IPM          0x00000F00
#define V_PCH_SATA_AHCI_PXSCTL_IPM_0        0x00000000
#define V_PCH_SATA_AHCI_PXSCTL_IPM_1        0x00000100
#define V_PCH_SATA_AHCI_PXSCTL_IPM_2        0x00000200
#define V_PCH_SATA_AHCI_PXSCTL_IPM_3        0x00000300
#define B_PCH_SATA_AHCI_PXSCTL_SPD          0x000000F0
#define V_PCH_SATA_AHCI_PXSCTL_SPD_0        0x00000000
#define V_PCH_SATA_AHCI_PXSCTL_SPD_1        0x00000010
#define V_PCH_SATA_AHCI_PXSCTL_SPD_2        0x00000020
#define V_PCH_SATA_AHCI_PXSCTL_SPD_3        0x00000030
#define B_PCH_SATA_AHCI_PXSCTL_DET          0x0000000F
#define V_PCH_SATA_AHCI_PXSCTL_DET_0        0x00000000
#define V_PCH_SATA_AHCI_PXSCTL_DET_1        0x00000001
#define V_PCH_SATA_AHCI_PXSCTL_DET_4        0x00000004
#define R_PCH_SATA_AHCI_P0SERR              0x130
#define R_PCH_SATA_AHCI_P1SERR              0x1B0
#define R_PCH_SATA_AHCI_P2SERR              0x230
#define R_PCH_SATA_AHCI_P3SERR              0x2B0
#define R_PCH_SATA_AHCI_P4SERR              0x330
#define R_PCH_SATA_AHCI_P5SERR              0x3B0
#define R_PCH_SATA_AHCI_P6SERR              0x430
#define B_PCH_SATA_AHCI_PXSERR_EXCHG        BIT26
#define B_PCH_SATA_AHCI_PXSERR_UN_FIS_TYPE  BIT25
#define B_PCH_SATA_AHCI_PXSERR_TRSTE_24     BIT24
#define B_PCH_SATA_AHCI_PXSERR_TRSTE_23     BIT23
#define B_PCH_SATA_AHCI_PXSERR_HANDSHAKE    BIT22
#define B_PCH_SATA_AHCI_PXSERR_CRC_ERROR    BIT21
#define B_PCH_SATA_AHCI_PXSERR_10B8B_DECERR BIT19
#define B_PCH_SATA_AHCI_PXSERR_COMM_WAKE    BIT18
#define B_PCH_SATA_AHCI_PXSERR_PHY_ERROR    BIT17
#define B_PCH_SATA_AHCI_PXSERR_PHY_RDY_CHG  BIT16
#define B_PCH_SATA_AHCI_PXSERR_INTRNAL_ERR  BIT11
#define B_PCH_SATA_AHCI_PXSERR_PROTOCOL_ERR BIT10
#define B_PCH_SATA_AHCI_PXSERR_PCDIE        BIT9
#define B_PCH_SATA_AHCI_PXSERR_TDIE         BIT8
#define B_PCH_SATA_AHCI_PXSERR_RCE          BIT1
#define B_PCH_SATA_AHCI_PXSERR_RDIE         BIT0
#define R_PCH_SATA_AHCI_P0SACT              0x134
#define R_PCH_SATA_AHCI_P1SACT              0x1B4
#define R_PCH_SATA_AHCI_P2SACT              0x234
#define R_PCH_SATA_AHCI_P3SACT              0x2B4
#define R_PCH_SATA_AHCI_P4SACT              0x334
#define R_PCH_SATA_AHCI_P5SACT              0x3B4
#define R_PCH_SATA_AHCI_P6SACT              0x434
#define B_PCH_SATA_AHCI_PXSACT_DS           0xFFFFFFFF
#define R_PCH_SATA_AHCI_P0CI                0x138
#define R_PCH_SATA_AHCI_P1CI                0x1B8
#define R_PCH_SATA_AHCI_P2CI                0x238
#define R_PCH_SATA_AHCI_P3CI                0x2B8
#define R_PCH_SATA_AHCI_P4CI                0x338
#define R_PCH_SATA_AHCI_P5CI                0x3B8
#define R_PCH_SATA_AHCI_P6CI                0x438
#define B_PCH_SATA_AHCI_PXCI                0xFFFFFFFF

//
//  SATA AHCI Device ID macros
//
#define IS_PCH_H_SATA_AHCI_DEVICE_ID(DeviceId) \
    ( \
      (DeviceId == V_SKL_PCH_H_SATA_DEVICE_ID_D_AHCI_A0) || \
      (DeviceId == V_SKL_PCH_H_SATA_DEVICE_ID_D_AHCI) || \
      (DeviceId == V_KBL_PCH_H_SATA_DEVICE_ID_D_AHCI) \
    )

#define IS_PCH_LP_SATA_AHCI_DEVICE_ID(DeviceId) \
    ( \
      (DeviceId == V_PCH_LP_SATA_DEVICE_ID_M_AHCI) \
    )

#define IS_PCH_SATA_AHCI_DEVICE_ID(DeviceId) \
    ( \
      IS_PCH_H_SATA_AHCI_DEVICE_ID(DeviceId) || \
      IS_PCH_LP_SATA_AHCI_DEVICE_ID(DeviceId) \
    )

//
//  SATA RAID Device ID macros
//
#define IS_PCH_H_SATA_RAID_DEVICE_ID(DeviceId) \
    ( \
      (DeviceId == V_SKL_PCH_H_SATA_DEVICE_ID_D_RAID) || \
      (DeviceId == V_SKL_PCH_H_SATA_DEVICE_ID_D_RAID_PREM) || \
      (DeviceId == V_PCH_H_SATA_DEVICE_ID_D_RAID_ALTDIS) || \
      (DeviceId == V_PCH_H_SATA_DEVICE_ID_D_RAID_RSTE) || \
      (DeviceId == V_SKL_PCH_H_SATA_DEVICE_ID_D_RAID_RRT) || \
      (DeviceId == V_KBL_PCH_H_SATA_DEVICE_ID_D_RAID) || \
      (DeviceId == V_KBL_PCH_H_SATA_DEVICE_ID_D_RAID_PREM) \
    )


#define IS_PCH_LP_SATA_RAID_DEVICE_ID(DeviceId) \
    ( \
      (DeviceId == V_PCH_LP_SATA_DEVICE_ID_M_RAID) || \
      (DeviceId == V_PCH_LP_SATA_DEVICE_ID_M_RAID_ALTDIS) || \
      (DeviceId == V_PCH_LP_SATA_DEVICE_ID_M_RAID_PREM) || \
      (DeviceId == V_PCH_LP_SATA_DEVICE_ID_M_RAID_RRT) \
    )

#define IS_PCH_SATA_RAID_DEVICE_ID(DeviceId) \
    ( \
      IS_PCH_H_SATA_RAID_DEVICE_ID(DeviceId) || \
      IS_PCH_LP_SATA_RAID_DEVICE_ID(DeviceId) \
    )

//
//  Combined SATA IDE/AHCI/RAID Device ID macros
//
#define IS_PCH_H_SATA_DEVICE_ID(DeviceId) \
    ( \
      IS_PCH_H_SATA_AHCI_DEVICE_ID(DeviceId) || \
      IS_PCH_H_SATA_RAID_DEVICE_ID(DeviceId) \
    )

#define IS_PCH_LP_SATA_DEVICE_ID(DeviceId) \
    ( \
      IS_PCH_LP_SATA_AHCI_DEVICE_ID(DeviceId) || \
      IS_PCH_LP_SATA_RAID_DEVICE_ID(DeviceId) \
    )
#define IS_PCH_SATA_DEVICE_ID(DeviceId) \
    ( \
      IS_PCH_H_SATA_DEVICE_ID(DeviceId) || \
      IS_PCH_LP_SATA_DEVICE_ID(DeviceId) \
    )

#endif
